package bmserv.model;

import java.util.GregorianCalendar;

import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlType;

/**
 * Bean JAXB qui implemente le modele de Tagging.
 * Un {@link Tagging} correspond &agrave; une action de tagging. <br/>
 * C'est &agrave; dire au moment pr&eacute;cis o&ugrave; un utilisateur associe un {@link Tag} avec un {@link Bookmark}. 
 * @author ndelafor
 *
 */
@XmlAccessorType(XmlAccessType.FIELD)
@XmlType(name="tagging", propOrder={
		"id",
		"authorid",
		"tagid",
		"bmid",
		"date"
	}
)
@XmlRootElement(name = "tagging")
public class Tagging {
	@XmlElement(name = "id")
	private String id;
	@XmlElement(name = "tagid")
	private String tagid;
	@XmlElement(name = "bmid")
	private String bmid;
	@XmlElement(name = "authorid")
	private String authorid;
	@XmlElement(name = "date")
	private String date;

	/**
	 * Constructeur vide pour JAXB.
	 */
	public Tagging() {
	}

	/**
	 * Creation complete d'un tagging.
	 * @param date chaine au format {@link GregorianCalendar} contenant la date a laquelle le {@link Tag} a ete associe au {@link Bookmark}.
	 * @param authorid ID du {@link User} a l'origine de l'association Tag/Bookmark
	 * @param bmid ID du {@link Bookmark}
	 * @param tagid ID du {@link Tag}
	 */
	public Tagging(final String date, final String authorid, final String bmid, final String tagid) {
		this.date = date;
		this.tagid = tagid;
		this.bmid = bmid;
		this.authorid = authorid;
		this.id = createTaggingId(tagid, bmid);
	}

	/**
	 * Getter sur l'ID du Tagging
	 * @return l'ID du Tagging
	 */
	public String getId() {
		return id;
	}

	/**
	 * Getter sur l'ID du Tag
	 * @return l'ID du Tag
	 */
	public String getTagId() {
		return tagid;
	}

	/**
	 * Getter sur l'ID du Bookmark
	 * @return l'ID du Bookmark
	 */
	public String getBmId() {
		return bmid;
	}

	/**
	 * Getter sur l'ID de l'auteur
	 * @return l'ID de l'auteur
	 */
	public String getAuthorId() {
		return authorid;
	}

	/**
	 * Getter sur la date a laquelle le tagging a ete cree
	 * @return une chaine au format {@link GregorianCalendar} contenant la date a laquelle le tagging a ete cree.
	 */
	public String getDate() {
		return date;
	}

	public void setTagId(final String tagid) {
		this.tagid = tagid;
		this.id = createTaggingId(this.tagid, this.bmid);
	}

	public void setBmId(final String bmid) {
		this.bmid = bmid;
		this.id = createTaggingId(this.tagid, this.bmid);
	}

	public void setAuthorId(final String authorid) {
		this.authorid = authorid;
	}

	/**
	 * Methode statique de generation d'un ID de tagging a partir d'un ID de {@link Tag} et d'un ID de {@link Bookmark}.<br/>
	 * Sachant que l'ID d'un {@link Bookmark} est deja compose d'un ID de {@link User} et de la signature MD5 de l'URL Bookmarkee 
	 * (voir methode {@link Bookmark Bookmark.createBmId()}) et qu'un ID de {@link Tag} est genere a partir de son label, 
	 * un ID de tagging devrait ressembler a : <code>(tagid)@(bmid)</code>
	 * @param tagid
	 * @param bmid
	 * @return
	 */
	public static String createTaggingId(final String tagid, final String bmid) {
		return tagid + "@" + bmid;
	}
}
