package bmserv.model.handlers;

import java.util.ArrayList;
import java.util.List;

import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;

import bmserv.Server;
import bmserv.Writer;
import bmserv.model.Bookmark;
import bmserv.model.Tag;
import bmserv.model.Tagging;

/**
 * Gestion des instances de {@link Bookmark}.
 * {@link BookmarkHandler} implemente les methodes CRUD sur les {@link Bookmark}.
 * @author ndelafor 
 */

@XmlAccessorType(XmlAccessType.FIELD)
@XmlRootElement(name = "bookmarks")
public class BookmarkHandler {
	@XmlElement(name = "bookmark", required=true)
	private static List<Bookmark> bookmarks = new ArrayList<Bookmark>();
	
	/**
	 * Constructeur vide pour JAXB
	 */
	public BookmarkHandler() {
	}

	/**
	 * Creation d'un nouveau {@link Bookmark} et generation d'un nouvel ID a partir des parametres.
	 * @param title le titre du {@link Bookmark}
	 * @param urlstr l'URL bookmark&eacute;e
	 * @param descr un petit texte descriptif
	 * @param authid l'ID de l'auteur du {@link Bookmark}
	 */
	public void createBookmark(String title, String urlstr, String descr, String login){
		String bmid = Bookmark.createBmId(login, urlstr);
		if(getBookmarkFromId(bmid)==null) {
			bookmarks.add(new Bookmark(title, urlstr, descr, login));
			Writer.serializeBookmarks();
		}
	}
	
	/**
	 * Renvoie tous les {@link Bookmark Bookmarks}
	 * @return une {@link List} de {@link Bookmark Bookmarks}
	 */
	public List<Bookmark> getBookmarks() {
		return bookmarks;
	}
	
	/**
	 * Permet de r&eacute;cup&eacute;rer un {@link Bookmark} &agrave; partir de son ID.
	 * @param bmid l'ID du {@link Bookmark} &agrave; r&eacute;cup&eacute;rer
	 * @return le {@link Bookmark}
	 */
	public Bookmark getBookmarkFromId(final String bmid) {
		for(Bookmark bm : bookmarks){
			if(bm.getId().equals(bmid)) return bm;
		}
		return null;
	}

	/**
	 * Permet de r&eacute;cup&eacute;rer la liste des {@link Bookmark Bookmarks} qui sont tagg&eacute;s avec un tagid.
	 * @param tagid l'ID du {@link Tag} recherch&eacute;
	 * @return Liste de Bookmarks
	 */
	public List<Bookmark> getBookmarkFromTagId(final String tagid) {
		List<Bookmark> blist = new ArrayList<Bookmark>();
		List<Tagging> tglist=Server.tgh.getTaggingsFromTagId(tagid);
		for(Tagging t : tglist){
			blist.add(getBookmarkFromId(t.getBmId()));
		}
		return blist;
	}
	

	/**
	 * Mise a jour "massive" d'un {@link Bookmark} existant &agrave; partir de son id.
	 * @param bmid l'ID du {@link Bookmark} &agrave; mettre &agrave; jour
	 * @param title le nouveau titre
	 * @param urlstr la nouvelle URL
	 * @param descr la nouvelle description
	 */
	public void updateBookmark(final String bmid, final String title, final String urlstr, final String descr){
		Bookmark bkmk = getBookmarkFromId(bmid);
		bkmk.setTitle(title);
		bkmk.setUrlstr(urlstr);
		bkmk.setDescription(descr);
		Server.tgh.updateTaggingsWithBookmarkId(bmid, bkmk.getId());
		Writer.serializeBookmarks();
	}
	
	/**
	 * Supprime un {@link Bookmark} de la liste.
	 * @param bmid
	 */
	public void removeBookmark(String bmid){
		Server.tgh.removeTaggingsWithBookmarkId(bmid);
		bookmarks.remove(getBookmarkFromId(bmid));
		Writer.serializeBookmarks();
	}
	
	/**
	 * Methode utilitaire pour afficher le nombre d'&eacute;l&eacute;ments contenus dans la liste.
	 */
	public void print(){
		System.out.print("=> "+bookmarks.size()+" éléments chargés\n");
	}	
}