package bmserv.model;

import java.util.ArrayList;
import java.util.List;

import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlType;

import bmserv.Server;
import bmserv.Utils;
import bmserv.model.handlers.BookmarkHandler;

/**
 * Bean JAXB qui implemente le modele de Bookmark.<br/>
 * Un {@link Bookmark} est une r&eacute;f&eacute;rence explicite cr&eacute;&eacute;e par un auteur vers une URL &agrave; laquelle sont associ&eacute;s un itre, une description.
 * @author ndelafor
 */
@XmlAccessorType(XmlAccessType.FIELD)
@XmlType(name = "bookmark", propOrder = { "id", "title", "urlstr", "authid", "description" })
@XmlRootElement(name = "bookmark")
public class Bookmark {
	@XmlElement(name = "id")
	private String id;
	@XmlElement(name = "title")
	private String title;
	@XmlElement(name = "description")
	private String description;
	@XmlElement(name = "urlstr")
	private String urlstr;
	@XmlElement(name = "authid")
	private String authid;

	/**
	 * Constructeur vide. Utilis&eacute; par JAXB.
	 */
	public Bookmark() {	}

	/**
	 * Constructeur param&eacute;tr&eacute; pour l'ajout dynamique de Bookmarks depuis le {@link BookmarkHandler}.
	 * @param title : le titre du {@link Bookmark}
	 * @param urlstr : l'URL bookmark&eacute;e
	 * @param descr : la description du {@link Bookmark}
	 * @param authid : l'ID de l'auteur du {@link Bookmark}
	 */
	public Bookmark(final String title, final String urlstr, final String descr, final String authid) {
		this.title = title;
		this.urlstr = urlstr;
		this.authid = authid;
		this.description = descr;
		this.id = createBmIdFormUserId(authid, urlstr);
	}

	/**
	 * Getter sur l'ID du bookmark
	 * @return l'ID du bookmark
	 */
	public String getId() {
		return id;
	}

	/**
	 * Getter sur le titre du bookmark
	 * @return le titre du bookmark
	 */
	public String getTitle() {
		return title;
	}

	/**
	 * Getter sur la description du bookmark
	 * @return la description du bookmark
	 */
	public String getDescription() {
		return description;
	}
	
	/**
	 * Getter sur l'URL du bookmark
	 * @return l'URL du bookmark
	 */
	public String getUrlstr() {
		return urlstr;
	}

	/**
	 * Getter sur l'id de l'auteur du bookmark.
	 * @return l'id de l'auteur du bookmark
	 */
	public String getAuthid() {
		return authid;
	}

	/**
	 * Getter sur l'objet {@link User} qui contient l'auteur du bookmark.
	 * @return le {@link User} qui contient l'auteur du bookmark.
	 */
	public User getAuthor() {
		return Server.uh.getUserFromId(this.authid);
	}

	/**
	 * R&eacute;cup&egrave;re tous les {@link Tag} associ&eacute;s au bookmark.
	 * @return la liste des tags
	 */
	public List<Tag> getAssociatedTags(){
		List<Tagging> glist = Server.tgh.getTaggingsFromBookmarkId(this.id);
		List<Tag> tlist = new ArrayList<Tag>();
		for(Tagging t : glist){
			tlist.add(Server.th.getTagFromId(t.getTagId()));
		}
		return tlist;
	}
	
	// SETTERS : pas de setter pour bmid ni pour authid qui ne doivent jamais changer
	
	/**
	 * Setter. Mise a jour du titre du bookmark
	 */
	public void setTitle(final String title) {
		this.title = title;
	}

	/**
	 * Setter. Mise a jour de la description du bookmark
	 */
	public void setDescription(final String description) {
		this.description = description;
	}

	/**
	 * Setter. Mise a jour de l'URL du bookmark
	 */
	public void setUrlstr(final String urlstr) {
		this.urlstr = urlstr;
		this.id = createBmIdFormUserId(this.authid, urlstr);
	}

	/**
	 * Methode statique de creation d'un ID de {@link Bookmark}.<br/>
	 * Les ID de {@link Bookmark} sont g&eacute;n&eacute;r&eacute;s &agrave; partir de l'ID de l'auteur et de la signature MD5 de l'URL.<br/>
	 * Par exemple : si mbuffa a bookmark&eacute; <a href="http://wiigen.fr/">http://wiigen.fr/</a>, l'ID sera : <br/>
	 * 	<code>URLEncoder.encode("mbuffa", "UTF-8") + "@" + MD5.encode("http://wiigen.fr/");</code>
	 * @param login le login de l'auteur du bookmark
	 * @param urlstr l'URL bookmark&eacute;e
	 * @return string contenant l'ID du bookmark
	 */
	public static String createBmId(final String login, final String urlstr) {
		String str = "";
		str = User.createUserId(login) + "@" + Utils.md5Encode(urlstr);
		return str;
	}
	public static String createBmIdFormUserId(final String userid, final String urlstr) {
		String str = "";
		str = userid + "@" + Utils.md5Encode(urlstr);
		return str;
	}
}
